import maya.cmds as cmds
import maya.mel as mel

jobId = None

def createMenuForModelPanel(modelPanel):
	try:
		viewMenu = cmds.menuBarLayout(modelPanel, query=True, menuArray=True, fullPathName=True)[0]
	except:
		return
	
	postMenuCmd = cmds.menu(f"{modelPanel}|{viewMenu}", q=True, postMenuCommand=True)
	mel.eval(postMenuCmd)

	viewMenuItems = cmds.menu(f"{modelPanel}|{viewMenu}", q=True, itemArray=True)
	beforeImagePlaneItem = None
	for idx, item in enumerate(viewMenuItems):
		if "mediaplanesubmenu" in item.lower():
			return
		
		if "imageplane" in item.lower():
			beforeImagePlaneItem = viewMenuItems[idx - 1]

	mpSubmenu = cmds.menuItem(
		f"mediaPlaneSubmenu",
		label="Media Plane",
		parent=f"{modelPanel}|{viewMenu}",
		insertAfter=beforeImagePlaneItem,
		subMenu=True,
		tearOff=True
	)

	_ = cmds.menuItem(
		f"mediaPlane_CreateNew",
		label="Create new...",
		parent=mpSubmenu,
		command=lambda *_: mediaPlaneCreateNew(modelPanel)
	)
	_ = cmds.menuItem(
		f"mediaPlane_SelectNode",
		label="Select node",
		parent=mpSubmenu,
		command=lambda *_: mediaPlaneSelectNode(modelPanel)
	)
	_ = cmds.menuItem(
		f"mediaPlane_AttachCamera",
		label="Attach selected to camera",
		parent=mpSubmenu,
		command=lambda *_: mediaPlaneAttachSelected(modelPanel)
	)
	_ = cmds.menuItem(
		f"mediaPlane_SetFile",
		label="Set media file...",
		parent=mpSubmenu,
		command=lambda *_: mediaPlaneSetFile(modelPanel)
	)
	_ = cmds.menuItem(
		f"mediaPlane_Delete",
		label="Delete",
		parent=mpSubmenu,
		command=lambda *_: mediaPlaneDelete(modelPanel)
	)

def createMenuForAllModelPanels():
	modelPanels = cmds.getPanel(type='modelPanel') or []
	for modelPanel in modelPanels:
		createMenuForModelPanel(modelPanel)

def destroyMenuForModelPanel(modelPanel):
	try:
		viewMenu = cmds.menuBarLayout(modelPanel, query=True, menuArray=True, fullPathName=True)[0]
	except:
		return
	
	postMenuCmd = cmds.menu(f"{modelPanel}|{viewMenu}", q=True, postMenuCommand=True)
	mel.eval(postMenuCmd)

	viewMenuItems = cmds.menu(f"{modelPanel}|{viewMenu}", q=True, itemArray=True)
	for item in viewMenuItems:
		if "mediaplanesubmenu" in item.lower():
			cmds.deleteUI(item, menuItem=True)
			return

def destroyMenuForAllModelPanels():
	modelPanels = cmds.getPanel(type='modelPanel') or []
	for modelPanel in modelPanels:
		destroyMenuForModelPanel(modelPanel)

def getModelPanelCameraShape(modelPanel):
	camera = cmds.modelPanel(modelPanel, q=True, camera=True)
	if not camera:
		return
	
	camShapes = cmds.listRelatives(camera, shapes=True)
	if not camShapes:
		return
	
	return camShapes[0]

def getMediaPlaneForCamera(camera, shape=True):
	camConns = cmds.listConnections(camera, s=True, d=True, shapes=True)
	if not camConns:
		return
	
	mps = cmds.ls(camConns, type="MediaPlane")
	if not mps:
		return
	
	if not shape:
		mpTransform = cmds.listRelatives(mps[0], parent=True)
		if not mpTransform:
			return
		
		return mpTransform[0]
	return mps[0]

def selectMediaFile():
	filters = "Media Files (*.mov *.mp4 *.avi *.3gp *.mkv *.webm *.mpeg *.flv *.png *.jpeg *.jpg *.exr *.tif *.tiff *.bmp);;Video Files (*.mov *.mp4 *.avi *.3gp *.mkv *.webm *.mpeg *.flv);;Image Files (*.png *.jpeg *.jpg *.exr *.tif *.tiff *.bmp);;All Files (*.*)";
	files = cmds.fileDialog2(fileMode=1, caption="Select Media File", fileFilter=filters) or [""]
	return files[0]

def mediaPlaneCreateNew(modelPanel):
	mpKwargs = {"c": getModelPanelCameraShape(modelPanel)}
	if mpKwargs["c"] is None:
		cmds.error(f"Couldn't get camera for viewport: {modelPanel}", noContext=True)
	
	mediaFile = selectMediaFile()
	if mediaFile:
		mpKwargs["f"] = mediaFile
	cmds.mediaPlane(**mpKwargs)

def mediaPlaneSelectNode(modelPanel):
	cameraShape = getModelPanelCameraShape(modelPanel)
	if cameraShape is None:
		cmds.error(f"Couldn't get camera for viewport: {modelPanel}", noContext=True)
	
	mp = getMediaPlaneForCamera(cameraShape, shape=False)
	if not mp:
		cmds.error(f"Couldn't get Media Plane for camera: {cameraShape}", noContext=True)
	cmds.select(mp)

def mediaPlaneAttachSelected(modelPanel):
	mediaPlanes = cmds.ls(sl=True, type="MediaPlane")
	if not mediaPlanes:
		mpTransforms = cmds.ls(sl=True, transforms=True)
		mediaPlanes = cmds.listRelatives(mpTransforms, shapes=True, type="MediaPlane")
	
	if not mediaPlanes:
		cmds.error(f"No Media Plane node is selected", noContext=True)

	cameraShape = getModelPanelCameraShape(modelPanel)
	if cameraShape is None:
		cmds.error(f"Couldn't get camera for viewport: {modelPanel}", noContext=True)

	attrs = {
		"filmFit": "cameraFilmFit",
		"focalLength": "cameraFocalLength",
		"horizontalFilmAperture": "cameraHAperture",
		"verticalFilmAperture": "cameraVAperture",
		"worldMatrix[0]": "cameraWorldMatrix"
	}
	for mediaPlane in mediaPlanes:
		for srcAttr, dstAttr in attrs.items():
			cmds.connectAttr(f"{cameraShape}.{srcAttr}", f"{mediaPlane}.{dstAttr}", f=True)

def mediaPlaneSetFile(modelPanel):
	cameraShape = getModelPanelCameraShape(modelPanel)
	if cameraShape is None:
		cmds.error(f"Couldn't get camera for viewport: {modelPanel}", noContext=True)
	
	mp = getMediaPlaneForCamera(cameraShape, shape=False)
	if not mp:
		mediaPlaneCreateNew(modelPanel)
		return
	
	mediaFile = selectMediaFile()
	if mediaFile:
		cmds.setAttr(f"{mp}.file", mediaFile, type="string")

def mediaPlaneDelete(modelPanel):
	cameraShape = getModelPanelCameraShape(modelPanel)
	if cameraShape is None:
		cmds.error(f"Couldn't get camera for viewport: {modelPanel}", noContext=True)
	
	mp = getMediaPlaneForCamera(cameraShape, shape=False)
	if not mp:
		cmds.error(f"Couldn't get Media Plane for camera: {cameraShape}", noContext=True)
	cmds.delete(mp)

def init():
	global jobId
	createMenuForAllModelPanels()
	jobId = cmds.scriptJob(event=["modelEditorChanged", createMenuForAllModelPanels], protected=True)

def destroy():
	global jobId
	if jobId is not None:
		cmds.scriptJob(kill=jobId, force=True)
	destroyMenuForAllModelPanels()
